<?php
/**
 * Admin dashboard: licenses, search, revoke, block domain
 */
require __DIR__ . '/includes/bootstrap.php';
require_admin();

// Actions
$message = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $id     = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $domain = isset($_POST['domain']) ? trim((string) $_POST['domain']) : '';

    if ($action === 'revoke' && $id > 0) {
        $stmt = $pdo->prepare('UPDATE licenses SET status = \'revoked\' WHERE id = ?');
        $stmt->execute([$id]);
        $message = 'License revoked.';
    }
    if ($action === 'block_domain' && $domain !== '') {
        $domain = preg_replace('/^https?:\/\//i', '', strtolower(explode('/', $domain)[0]));
        if (preg_match('/^[a-z0-9]([a-z0-9\-\.]*[a-z0-9])?$/i', $domain)) {
            $stmt = $pdo->prepare('INSERT IGNORE INTO blocked_domains (domain, reason) VALUES (?, ?)');
            $stmt->execute([$domain, 'Blocked from admin']);
            $message = $stmt->rowCount() ? 'Domain blocked.' : 'Domain already blocked or invalid.';
        }
    }
}

// Counts
$totalActive = (int) $pdo->query("SELECT COUNT(*) FROM licenses WHERE status = 'active'")->fetchColumn();
$totalAll    = (int) $pdo->query('SELECT COUNT(*) FROM licenses')->fetchColumn();

// Search
$search = isset($_GET['q']) ? trim((string) $_GET['q']) : '';
$licenses = [];
if ($search !== '') {
    $stmt = $pdo->prepare("SELECT id, domain, email, status, activation_date, last_check, ip_address FROM licenses WHERE domain LIKE ? OR email LIKE ? OR purchase_code_hash LIKE ? ORDER BY id DESC LIMIT 200");
    $stmt->execute(["%$search%", "%$search%", "%$search%"]);
} else {
    $stmt = $pdo->query("SELECT id, domain, email, status, activation_date, last_check, ip_address FROM licenses ORDER BY id DESC LIMIT 100");
}
$licenses = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Blocked domains for display
$blocked = $pdo->query('SELECT domain, reason, blocked_at FROM blocked_domains ORDER BY blocked_at DESC LIMIT 50')->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>License Admin – Dashboard</title>
    <style>
        * { box-sizing: border-box; }
        body { font-family: system-ui, sans-serif; background: #1a1d24; color: #e2e4e8; margin: 0; min-height: 100vh; }
        .header { background: #252830; padding: 1rem 1.5rem; display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 1rem; }
        .header h1 { margin: 0; font-size: 1.25rem; }
        .header a { color: #7dd3fc; text-decoration: none; }
        .header a:hover { text-decoration: underline; }
        .container { max-width: 1200px; margin: 0 auto; padding: 1.5rem; }
        .cards { display: grid; grid-template-columns: repeat(auto-fill, minmax(180px, 1fr)); gap: 1rem; margin-bottom: 1.5rem; }
        .card { background: #252830; padding: 1.25rem; border-radius: 8px; }
        .card .num { font-size: 1.75rem; font-weight: 700; color: #7dd3fc; }
        .card .label { font-size: 0.85rem; color: #94a3b8; }
        .toolbar { margin-bottom: 1rem; display: flex; gap: 0.75rem; flex-wrap: wrap; align-items: center; }
        .toolbar input { padding: 0.5rem 0.75rem; border: 1px solid #3d424d; border-radius: 4px; background: #252830; color: #e2e4e8; }
        .toolbar button, .btn { padding: 0.5rem 1rem; border: none; border-radius: 4px; cursor: pointer; font-size: 0.9rem; text-decoration: none; display: inline-block; }
        .btn-primary { background: #0d6efd; color: #fff; }
        .btn-primary:hover { background: #0b5ed7; }
        .btn-danger { background: #dc3545; color: #fff; }
        .btn-danger:hover { background: #bb2d3b; }
        .message { background: #14532d; color: #86efac; padding: 0.75rem; border-radius: 4px; margin-bottom: 1rem; }
        table { width: 100%; border-collapse: collapse; background: #252830; border-radius: 8px; overflow: hidden; }
        th, td { padding: 0.75rem 1rem; text-align: left; border-bottom: 1px solid #3d424d; }
        th { background: #2d323d; font-size: 0.8rem; text-transform: uppercase; color: #94a3b8; }
        tr:hover { background: #2d323d; }
        .status-active { color: #86efac; }
        .status-revoked, .status-blocked { color: #f87171; }
        .section { margin-bottom: 2rem; }
        .section h2 { font-size: 1.1rem; margin-bottom: 0.75rem; }
        .block-form { display: flex; gap: 0.5rem; margin-bottom: 1rem; }
        .block-form input { flex: 1; max-width: 300px; padding: 0.5rem; border: 1px solid #3d424d; border-radius: 4px; background: #1a1d24; color: #e2e4e8; }
    </style>
</head>
<body>
    <header class="header">
        <h1>Smart SEO Audit – License Server</h1>
        <div>
            <span><?php echo htmlspecialchars($_SESSION['admin_username'] ?? ''); ?></span>
            <a href="logout.php">Log out</a>
        </div>
    </header>
    <div class="container">
        <?php if ($message): ?><p class="message"><?php echo htmlspecialchars($message); ?></p><?php endif; ?>

        <div class="cards">
            <div class="card">
                <div class="num"><?php echo $totalActive; ?></div>
                <div class="label">Active licenses</div>
            </div>
            <div class="card">
                <div class="num"><?php echo $totalAll; ?></div>
                <div class="label">Total activations</div>
            </div>
        </div>

        <div class="section">
            <h2>Search license</h2>
            <form method="get" class="toolbar">
                <input type="text" name="q" placeholder="Domain, email, or hash..." value="<?php echo htmlspecialchars($search); ?>">
                <button type="submit" class="btn btn-primary">Search</button>
            </form>
        </div>

        <div class="section">
            <h2>Block domain</h2>
            <form method="post" class="block-form">
                <input type="hidden" name="action" value="block_domain">
                <input type="text" name="domain" placeholder="example.com" required>
                <button type="submit" class="btn btn-danger">Block domain</button>
            </form>
        </div>

        <div class="section">
            <h2>Licenses</h2>
            <table>
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Domain</th>
                        <th>Email</th>
                        <th>Status</th>
                        <th>Activation</th>
                        <th>Last check</th>
                        <th>IP</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($licenses as $r): ?>
                    <tr>
                        <td><?php echo (int) $r['id']; ?></td>
                        <td><?php echo htmlspecialchars($r['domain']); ?></td>
                        <td><?php echo htmlspecialchars($r['email']); ?></td>
                        <td><span class="status-<?php echo htmlspecialchars($r['status']); ?>"><?php echo htmlspecialchars($r['status']); ?></span></td>
                        <td><?php echo htmlspecialchars($r['activation_date'] ?? '-'); ?></td>
                        <td><?php echo htmlspecialchars($r['last_check'] ?? '-'); ?></td>
                        <td><?php echo htmlspecialchars($r['ip_address'] ?? '-'); ?></td>
                        <td>
                            <?php if ($r['status'] === 'active'): ?>
                            <form method="post" style="display:inline;" onsubmit="return confirm('Revoke this license?');">
                                <input type="hidden" name="action" value="revoke">
                                <input type="hidden" name="id" value="<?php echo (int) $r['id']; ?>">
                                <button type="submit" class="btn btn-danger">Revoke</button>
                            </form>
                            <?php else: ?>—<?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php if (empty($licenses)): ?><p>No licenses found.</p><?php endif; ?>
        </div>

        <div class="section">
            <h2>Blocked domains</h2>
            <table>
                <thead>
                    <tr>
                        <th>Domain</th>
                        <th>Reason</th>
                        <th>Blocked at</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($blocked as $b): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($b['domain']); ?></td>
                        <td><?php echo htmlspecialchars($b['reason'] ?? '-'); ?></td>
                        <td><?php echo htmlspecialchars($b['blocked_at'] ?? '-'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php if (empty($blocked)): ?><p>No blocked domains.</p><?php endif; ?>
        </div>
    </div>
</body>
</html>
