<?php
/**
 * Bootstrap: load config, DB, and security for API endpoints
 */

declare(strict_types=1);

if (!defined('LICENSE_SERVER')) {
    http_response_code(403);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Forbidden']);
    exit;
}

$configPath = __DIR__ . '/config.php';
if (!is_readable($configPath)) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Server configuration error']);
    exit;
}

$config = require $configPath;

// PDO
$dsn = sprintf(
    'mysql:host=%s;dbname=%s;charset=%s',
    $config['db']['host'],
    $config['db']['name'],
    $config['db']['charset']
);
try {
    $pdo = new PDO($dsn, $config['db']['user'], $config['db']['password'], [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
} catch (PDOException $e) {
    http_response_code(500);
    header('Content-Type: application/json');
    echo json_encode(['error' => 'Database unavailable']);
    exit;
}

function json_response(array $data, int $code = 200): void {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    header('X-Content-Type-Options: nosniff');
    echo json_encode($data, JSON_UNESCAPED_SLASHES);
    exit;
}

function get_client_ip(): string {
    $keys = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
    foreach ($keys as $key) {
        if (!empty($_SERVER[$key])) {
            $ip = trim(explode(',', (string) $_SERVER[$key])[0]);
            if (filter_var($ip, FILTER_VALIDATE_IP)) {
                return $ip;
            }
        }
    }
    return '0.0.0.0';
}

function validate_api_key(array $config): bool {
    $key = $_SERVER['HTTP_X_API_KEY'] ?? $_POST['api_key'] ?? $_GET['api_key'] ?? '';
    return is_string($key) && hash_equals($config['api_secret_key'], $key);
}

function check_rate_limit(PDO $pdo, array $config, string $endpoint): void {
    $ip = get_client_ip();
    $limit = (int) ($config['rate_limit'] ?? 60);
    $since = date('Y-m-d H:i:s', time() - 3600);

    $pdo->prepare('DELETE FROM rate_limit_log WHERE request_at < ?')->execute([$since]);

    $stmt = $pdo->prepare('SELECT COUNT(*) FROM rate_limit_log WHERE ip_address = ? AND endpoint = ? AND request_at > ?');
    $stmt->execute([$ip, $endpoint, $since]);
    $count = (int) $stmt->fetchColumn();

    if ($count >= $limit) {
        json_response(['error' => 'Rate limit exceeded', 'status' => 'invalid'], 429);
    }

    $pdo->prepare('INSERT INTO rate_limit_log (ip_address, endpoint) VALUES (?, ?)')->execute([$ip, $endpoint]);
}

function is_domain_blocked(PDO $pdo, string $domain): bool {
    $stmt = $pdo->prepare('SELECT 1 FROM blocked_domains WHERE domain = ? LIMIT 1');
    $stmt->execute([$domain]);
    return (bool) $stmt->fetch();
}

function hash_purchase_code(string $code): string {
    return hash('sha256', trim($code) . 'SSA_LICENSE_SALT');
}
