<?php
/**
 * Verify endpoint: validate Envato purchase code and activate license for domain
 * POST: purchase_code, domain, email (+ api_key or X-Api-Key header)
 */

declare(strict_types=1);

define('LICENSE_SERVER', true);

require __DIR__ . '/includes/bootstrap.php';

// Only allow POST; block direct browser GET
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    json_response(['error' => 'Method not allowed', 'status' => 'invalid'], 405);
}

if (!validate_api_key($config)) {
    json_response(['error' => 'Invalid API key', 'status' => 'invalid'], 403);
}

check_rate_limit($pdo, $config, 'verify');

$purchase_code = isset($_POST['purchase_code']) ? trim((string) $_POST['purchase_code']) : '';
$domain        = isset($_POST['domain']) ? trim((string) $_POST['domain']) : '';
$email         = isset($_POST['email']) ? trim((string) $_POST['email']) : '';

// Sanitize domain: allow only valid hostname
$domain = preg_replace('/^https?:\/\//i', '', $domain);
$domain = strtolower(explode('/', $domain)[0]);
if (!preg_match('/^[a-z0-9]([a-z0-9\-\.]*[a-z0-9])?$/i', $domain) || strlen($domain) > 255) {
    json_response(['error' => 'Invalid domain', 'status' => 'invalid'], 400);
}

$email = filter_var($email, FILTER_SANITIZE_EMAIL);
if (!filter_var($email, FILTER_VALIDATE_EMAIL) || strlen($email) > 255) {
    json_response(['error' => 'Invalid email', 'status' => 'invalid'], 400);
}

if (strlen($purchase_code) < 10) {
    json_response(['error' => 'Invalid purchase code', 'status' => 'invalid'], 400);
}

if (is_domain_blocked($pdo, $domain)) {
    json_response(['error' => 'Domain is blocked', 'status' => 'invalid'], 403);
}

$purchase_code_hash = hash_purchase_code($purchase_code);
$item_id = (string) $config['envato']['item_id'];
$token = $config['envato']['personal_token'];

// Check if this purchase code is already used for another domain
$stmt = $pdo->prepare('SELECT id, domain, status FROM licenses WHERE purchase_code_hash = ?');
$stmt->execute([$purchase_code_hash]);
$existing = $stmt->fetch();

if ($existing) {
    if ($existing['domain'] !== $domain) {
        json_response([
            'error'  => 'License already activated on another domain',
            'status' => 'invalid',
        ], 409);
    }
    if ($existing['status'] !== 'active') {
        json_response(['error' => 'License revoked or blocked', 'status' => 'invalid'], 403);
    }
    // Same domain, already active: update last_check and return valid
    $ip = get_client_ip();
    $pdo->prepare('UPDATE licenses SET last_check = NOW(), ip_address = ? WHERE id = ?')->execute([$ip, $existing['id']]);
    json_response(['status' => 'valid', 'message' => 'License already active']);
}

// Call Envato API
$url = 'https://api.envato.com/v3/market/author/sale?code=' . rawurlencode($purchase_code);
$ch = curl_init($url);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER     => ['Authorization: Bearer ' . $token],
    CURLOPT_TIMEOUT        => 15,
]);
$response = curl_exec($ch);
$httpCode = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
curl_close($ch);

if ($httpCode !== 200 || $response === false) {
    json_response(['error' => 'Could not verify purchase with Envato', 'status' => 'invalid'], 400);
}

$data = json_decode($response, true);
if (!is_array($data) || !isset($data['purchase'])) {
    json_response(['error' => 'Invalid purchase', 'status' => 'invalid'], 400);
}

$purchase = $data['purchase'];
$purchase_item_id = isset($purchase['item']['id']) ? (string) $purchase['item']['id'] : '';

if ($purchase_item_id !== $item_id) {
    json_response(['error' => 'Purchase is not for this product', 'status' => 'invalid'], 400);
}

$ip = get_client_ip();
$stmt = $pdo->prepare(
    'INSERT INTO licenses (purchase_code_hash, domain, email, status, ip_address) VALUES (?, ?, ?, \'active\', ?)'
);
$stmt->execute([$purchase_code_hash, $domain, $email, $ip]);

json_response(['status' => 'valid', 'message' => 'License activated successfully']);
